﻿import xbmc
import json
import os
import xbmcvfs
import re
import time
import random
from collections import OrderedDict

class AudioFolderManager(xbmc.Player):
    def __init__(self):
        super().__init__()
        self.storage_path = xbmcvfs.translatePath('special://profile/addon_data/service.audio.folder/db.json')
        self.max_entries = 100
        self._ensure_storage()
        self.db = self._load_db()
        self.last_index = -1
        self.current_file = ""
        self.ready_to_log = False
        self.session_id = 0

    def _ensure_storage(self):
        folder = os.path.dirname(self.storage_path)
        if not xbmcvfs.exists(folder): xbmcvfs.mkdirs(folder)
        if not xbmcvfs.exists(self.storage_path): 
            self._write_to_disk({})

    def _load_db(self):
        try:
            with xbmcvfs.File(self.storage_path, 'r') as f:
                return OrderedDict(json.loads(f.read()))
        except: return OrderedDict()

    def _write_to_disk(self, data):
        with xbmcvfs.File(self.storage_path, 'w') as f:
            f.write(json.dumps(data, indent=4, ensure_ascii=False))

    def _is_valid_path(self, file_path):
        if not file_path: return False
        fp_low = file_path.lower()
        is_net = fp_low.startswith(('smb://', 'nfs://', 'dav://', 'pfs://'))
        is_and = fp_low.startswith('/storage/')
        is_win = len(file_path) > 1 and file_path[1] == ':'
        is_loc = fp_low.startswith('/') and not is_and
        
        if not (is_net or is_and or is_win or is_loc): return False
        
        clean = re.sub(r'^[a-z]+://', '', file_path).replace('\\', '/').strip('/')
        parts = [p for p in clean.split('/') if p]
        
        if is_win: return len(parts) >= 3
        if is_and: return len(parts) >= 5
        if is_net: return len(parts) >= 4
        if is_loc: return len(parts) >= 2
        return False

    def _is_likely_episode(self, file_path):
        if not self._is_valid_path(file_path): return False
        filename = os.path.basename(file_path).lower()
        patterns = [r's\d+', r'e\d+', r'ep\d+', r'season', r'episode', r'сезон', r'серия', r'\d+x\d+']
        return any(re.search(p, filename) for p in patterns)

    def apply_settings(self):
        current_session = random.getrandbits(32)
        self.session_id = current_session
        self.ready_to_log = False
        
        xbmc.sleep(500)

        for _ in range(50):
            if self.isPlayingVideo() and self.getPlayingFile():
                if len(self.getAvailableAudioStreams()) > 0:
                    break
            xbmc.sleep(100)

        if self.session_id != current_session: return

        new_file = self.getPlayingFile()
        if not new_file or not self._is_likely_episode(new_file):
            return

        self.current_file = new_file
        folder = os.path.dirname(new_file)
        self.db = self._load_db()
        
        streams = self.getAvailableAudioStreams()
        if folder in self.db:
            target = self.db[folder]
            if target < len(streams):
                self.setAudioStream(target)
                self.last_index = target
            else:
                self.last_index = self.get_current_index()
        else:
            self.last_index = self.get_current_index()

        xbmc.sleep(1000)
        if self.session_id == current_session:
            self.ready_to_log = True

    def get_current_index(self):
        try:
            rpc = '{"jsonrpc":"2.0","method":"Player.GetProperties","params":[1,["currentaudiostream"]],"id":1}'
            res = json.loads(xbmc.executeJSONRPC(rpc))
            return res['result']['currentaudiostream']['index']
        except: return -1

    def onPlayBackStarted(self):
        self.apply_settings()

    def onAVStarted(self):
        if self.getPlayingFile() != self.current_file:
            self.apply_settings()

    def onAVChange(self):
        if not self.ready_to_log:
            return

        file_path = self.getPlayingFile()
        if not file_path or file_path != self.current_file:
            return

        xbmc.sleep(600)
        new_index = self.get_current_index()
        
        if new_index != -1 and new_index != self.last_index:
            folder = os.path.dirname(file_path)
            self.db = self._load_db()
            self.db[folder] = new_index
            self.db.move_to_end(folder)
            while len(self.db) > self.max_entries:
                self.db.popitem(last=False)
            self._write_to_disk(self.db)
            self.last_index = new_index

if __name__ == '__main__':
    monitor = xbmc.Monitor()
    player = AudioFolderManager()
    while not monitor.abortRequested():
        if monitor.waitForAbort(1): break